/******************************************************************************
 * Librairie: CEC_lib                                                         *
 *   Fichier: CEC_lib.cpp                                                     *
 *    Auteur: Laurent MARTIN                                                  *
 *                                                                            *
 *   Version  date        Description                                         *
 *   2.0      18/03/2022  Version opérationnelle                              *
 *   1.0      24/02/2022  Version initiale                                    *
 *                                                                            *
 ******************************************************************************/
/*******************************************************************************
  contient la définition de toute les fonctions utilisé pour controler le
  moteur par l'arduino.
  Fonctions presentes:
  - InitSoftserial(): permet d'initialiser le port serie TX, RX pour l'ensemble
    du projet
  - trameConfig(): fonction permettant d'envoyer la cartographie de niveau de
    vitesse au moteur ainsi que les infos d'une course (fonction a revoir pour
    utilisation simplifié dans mBlock
  - MotorStart(int temps): declenche le départ du moteur après un temps donné
  - MotorStop(int temps): declenche l'arret du moteur après un temps donné
  - MotorPause(int temps): declenche une pause du moteur pendant un certain
    temps
  - setspeed(int vitesse): regle coefficient permettant de calculé la vitesse en
    % de la vitesse max
  - moveForward(int temps,int Speed): declenche le moteur vers l'avant à une
    vitesse donnée et pendant un temps donné
  - moveBackward(int temps,int Speed): declenche le moteur vers l'arrière à une
    vitesse donnée et pendant un temps donné
  - les fonction get permettant de récuperer les infos prise par la carte moteur
    (a tester)
  - Convertintohex(): permet la traduction des valeurs en int vers de
    l'hexadecimal
*******************************************************************************/
#include <Arduino.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "CEC_lib.h"

#ifdef ARDUINO_DUE
// pour les cartes Arduino DUE ou les cartes ayant un deuxième port série :
// Choix du port série de sortie texte :
#define SerialOut Serial        // Port série de sortie Texte usb programmation
//#define SerialOut SerialUSB     // Port série de sortie Texte usb native
#define motorSerial Serial1

#else
// Pour les cartes Arduino UNO :
#define SerialOut Serial    // Port série de sortie Texte
#include <SoftwareSerial.h> // Port Série logiciel supplémentaire
SoftwareSerial motorSerial(10, 11); //Pin 10:RX  Pin 11:TX
#endif

// Sortie texte de debug
#define debugTxt  1 // sortie de debug texte: 0=sans, 1=avec

/*------------------Class Moteur------------------*/

/*******************************************************
 * Method: Moteur()
 * Constructeur
 *******************************************************/
Moteur::Moteur()
{
  memcpy(trameConfig.car, trameConfigInit, CEC_LENGTH_CONFIG);
}

/*******************************************************
 * Method: start()
 * Ouverture du port de communication moteur
 *******************************************************/
void Moteur::start(uint32_t serialConfig)
{
  motorSerial.begin(serialConfig); //ouverture du port série
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    if(SerialOut)
      #ifdef ARDUINO_DUE
      if (SerialOut.availableForWrite())
      #endif
        SerialOut.println(F("Connex. Moteur activée."));
  }
  #endif
}

/*******************************************************
 * Method: setEchoOn()
 * active le mode echo text vers sortie serie console
 *******************************************************/
void Moteur::setEchoOn()
{
  bEchoOn = false;  
}
    
/*******************************************************
 * Method: setEchoOff()
 * désactive le mode echo text vers sortie serie console
 *******************************************************/
void Moteur::setEchoOff()
{
  bEchoOn = true;  
}

/*******************************************************
 * Method: purge()
 * Vide le buffer de reception moteur
 *******************************************************/
void Moteur::purge()
{  
  #ifndef CODE_LIGHT
  if(bEchoOn)
    SerialOut.print(F("Purge connex. moteur:"));
  #endif    

  valUInt = 0;
  // Activation de la réception
  if(!motorSerial.isListening())
  {
    SerialOut.print(F("[NotListening]"));
    motorSerial.listen();
  }
  // Lecture des caractères tant qu'il y en a
  while(motorSerial.available())
  {
    motorSerial.read();
    valUInt++;
    delay(10);
  }
  #ifndef CODE_LIGHT
  if(bEchoOn)
    SerialOut.print(valUInt, DEC);
    SerialOut.println(F(" OK"));
  #endif
}

/*******************************************************
 * Method: writeHeader()
 *******************************************************/
int Moteur::writeHeader(byte cmd, uint16_t dataSize, bool bWithCheckSum)
{
  sendCmd = cmd;
  
  // Avec ou sans checkSum
  if(bWithCheckSum)
  {
    bCheckSum = true;
    checkSum = 0;
    this->trameHeader.addr = 0xA3; // avec crc
  }
  else
  {
    bCheckSum = false;
    this->trameHeader.addr = 0xA4; // pas de crc
  }

  //début de trame
  this->trameHeader.start = 0x02;
  
  // nb octets trame de config
  this->trameHeader.lengthMSB = (byte)(dataSize >> 8);
  this->trameHeader.lengthLSB = (byte)(dataSize & 0x00FF);

  /*trame de config*/
  this->trameHeader.cmd = sendCmd; //commande CONFIGURATION

  return CEC_ERR_OK;
}

/*******************************************************
 * Method: sendTrame()
 *******************************************************/
int Moteur::sendTrame(byte cmd, byte* ptData, uint16_t dataSize, bool bWithCheckSum)
{
  uint16_t i;
  byte* pt;

#ifdef ARDUINO_DUE
  if(!motorSerial.availableForWrite())
  {
    #ifndef CODE_LIGHT
    if(bEchoOn)
      SerialOut.println(F("connex. moteur non dispo (tx)"));
    #endif
        
    return CEC_ERR_SERIAL_NOT_AVAILABLE; // Serial not available
  }
#endif

  // Partie Header
  writeHeader(cmd, dataSize+1, bWithCheckSum);

  // Partie Footer: Calcul du crc et préparation du footer
  calculCrc(dataSize, ptData);
  writeFooter();
 
  // Envoi partie Header
  for(i = 0, pt = this->trameHeader.car; i < CEC_LENGTH_HEADER; i++)
  {
    motorSerial.write(*pt++);  
  }
  
  // Envoi partie Data
  if(dataSize > 0 && ptData)
  {
    for(i = 0, pt = ptData; i < dataSize; i++)
    {
      motorSerial.write(*pt++);  
    }
  }
  
  // Envoi partie Footer
  for(i = 0, pt = trameFooter; i < CEC_LENGTH_FOOTER; i++)
  {
    motorSerial.write(*pt++);  
  }

  return CEC_ERR_OK;
}

/*******************************************************
 * Method: writeFooter()
 *******************************************************/
int Moteur::writeFooter()
{
  byte *pt = trameFooter;

  // Crc de fin de trame
  *pt++ = (byte)(checkSum >> 8); // crc MSB
  *pt++ = (byte)(checkSum & 0x00FF); // crc LSB
  // fin de trame
  *pt++ = 0x03;

  return CEC_ERR_OK;
}

/*******************************************************
 * Method: calculCrc()
 * Calcule le crc du message avec header et footer
 *******************************************************/
int Moteur::calculCrc(uint16_t dataSize, byte* ptData)
{
  checkSum = 0;
  return 0;
}

/*******************************************************
 * Method: readHeader()
 *******************************************************/
int Moteur::readHeader()
{
  int t = CEC_TIMEOUT1; // Timeout de reception de caractères
  int id; // ID de caractère reçu
  byte *pt; // caractère reçu
  int err = CEC_ERR_OK;

  pt = this->trameHeader.car; // caractère reçu
  id = 0;
  this->longueurData = 0;

  // Activation de la réception
  if(!motorSerial.isListening())
  {
    SerialOut.print(F("[NotListening]"));
    motorSerial.listen();
  }
  // attente premier caractère pendant premier timeout
  while (!motorSerial.available() && t > 0)
  {
    delay(1);
    t--;
  }
  if(t == 0)
  {
    #ifndef CODE_LIGHT
    if(bEchoOn)
      SerialOut.print(F("readHeader: timeout1 après "));
      SerialOut.print(id);
      SerialOut.println(F(" car!"));
      if(!motorSerial.isListening())
      {
        SerialOut.print(F("[NotListening]"));
      }
    #endif    

    return CEC_ERR_TIMEOUT;
  }
  
  // réception caractères 
  t = CEC_TIMEOUT2;
  while(id < CEC_LENGTH_HEADER && t > 0)
  {
    while (!motorSerial.available() && t > 0)
    {
      delay(1);
      t--;
    }

    if(t == 0)
    {
      #ifndef CODE_LIGHT
      if(bEchoOn)
        SerialOut.println(F("readHeader: timeout2!"));
      #endif
              
      return CEC_ERR_TIMEOUT;
    }

    // Lecture des caractères
    *pt = motorSerial.read();
    pt++;
    switch(id)
    {
      case 0: // code message
        if (this->trameHeader.addr == 0xF0)
          return CEC_ERR_OK_F0;
        
        if (this->trameHeader.addr == 0xF1)
        {
          #ifndef CODE_LIGHT
          if(bEchoOn)
            SerialOut.println(F("readHeader: Ret. Erreur 0xF1"));
          #endif
          
          return CEC_ERR_KO_F1;
        }
        
        if (this->trameHeader.addr != 0xA4 && this->trameHeader.addr != 0xA3)
        {
          err = CEC_ERR;
        }
        id++;
        break;

      case 1: // début de trame
        if (this->trameHeader.start != 0x02)
        { 
          err = CEC_ERR;
        }
        id++;
        break;            

      case 2: // longueur MSB
        id++;
        break;

      case 3: // longueur LSB
        id++;
        this->longueurData = ((uint16_t) this->trameHeader.lengthMSB << 8) + this->trameHeader.lengthLSB;
        break;

      case 4: // commande
        this->rcvCmd = this->trameHeader.cmd;
        id++;
        break;

      default:
        err = CEC_ERR;
        break;
    }
  }
  return err;
}

/*******************************************************
 * Method: readData()
 *******************************************************/
int Moteur::readData(uint16_t nData, byte* ptData)
{
  int t = CEC_TIMEOUT3; // Timeout de reception de caractères
  uint16_t id = 0; // ID de caractère reçu
  byte *pt = ptData; // caractère reçu
  uint16_t val16; // valeur 16 bits
  int carId = CEC_LENGTH_HEADER;

  if(pt == NULL)  // pas de buffer passé
  {
    #ifndef CODE_LIGHT
    if(bEchoOn)
      SerialOut.println(F("readData: pas de buffer!"));
    #endif      
    return CEC_ERR;
  }

  // Activation de la réception
  if(!motorSerial.isListening())
  {
    SerialOut.print(F("[NotListening]"));
    motorSerial.listen();
  }
  // attente premier caractère pendant premier timeout
  while (!motorSerial.available() && t > 0)
  {
    delay(1);
    t--;
  }
  if(t == 0)
  {
    #ifndef CODE_LIGHT
    if(bEchoOn)
      SerialOut.println(F("readData: timeout3!"));
    #endif      
    
    return CEC_ERR_TIMEOUT;
  }

  // réception caractères 
  t = CEC_TIMEOUT5;
  while(id < nData && t > 0)
  {
    while (!motorSerial.available() && t > 0)
    {
      delay(1);
      t--;
    }

    if(t == 0)
    {
      #ifndef CODE_LIGHT
      if(bEchoOn)
      {
        SerialOut.print(F("readData: timeout5! "));
        SerialOut.print(id);
        SerialOut.println(F(" car."));
      }
      #endif
      
      return CEC_ERR_TIMEOUT;
    }    
    *pt = motorSerial.read();
    pt++;
    t = CEC_TIMEOUT5;
    id++;
  }
  return CEC_ERR_OK; 
}

/*******************************************************
 * Method: readFooter()
 *******************************************************/
int Moteur::readFooter()
{
  int t = CEC_TIMEOUT3; // Timeout de reception de caractères
  int id = 0; // ID de caractère reçu
  byte car; // caractère reçu
  uint16_t val16; // valeur 16 bits 
  int err = CEC_ERR_OK;

  // Activation de la réception
  if(!motorSerial.isListening())
  {
    SerialOut.print(F("[NotListening]"));
    motorSerial.listen();
  }
  // attente premier caractère pendant premier timeout
  while (!motorSerial.available() && t > 0)
  {
    delay(1);
    t--;
  }
  if(t == 0)
  {
    #ifndef CODE_LIGHT
    if(bEchoOn)
      SerialOut.println(F("readData: timeout3!"));
    #endif
    
    return CEC_ERR_TIMEOUT;
  }

  // réception caractères 
  t = CEC_TIMEOUT4;
  while(t > 0)
  {
    while (!motorSerial.available() && t > 0)
    {
      delay(1);
      t--;
    }

    if(t == 0)
    {
      if(id < CEC_LENGTH_FOOTER)
      {
        #ifndef CODE_LIGHT
        if(bEchoOn)
          SerialOut.println(F("readData: timeout4!"));
        #endif
        return CEC_ERR_TIMEOUT;
      }
      
      return err;
    }
    
    car = motorSerial.read();
    switch(id)
    {
      case 0: // checksum MSB
        val16 = (uint16_t) car << 8;
        id++;
        break;

      case 1: // checksum LSB
        val16 &= car;
        id++;
        break;

      case 2: // fin de trame
        if (car != 0x03)
          err = CEC_ERR;
        else if(bCheckSum && checkSum != val16)
          err = CEC_ERR_WRONG_CHECKSUM; // bonne reception mais mauvais checksum
        id++;
        if(err == CEC_ERR_OK)
          return err;
        break;

      default:
        break;
    }
  }
  return CEC_ERR_OK;
}

/*******************************************************
 * Method: sendConfig()
 *******************************************************/
int Moteur::sendConfig()
{
  int err = 0;
    
  // Set fixed values
  trameConfig.nDentsPignon = 12; //nb dents pignon moteur = 12
  trameConfig.nDentsCouronne = 34; //nb dents couronne = 34
  err = sendTrame(CEC_CMD_CONFIG, trameConfig.car, CEC_LENGTH_CONFIG, CEC_WITHOUT_CHECKSUM);
  if(err < 0)
    return err;
  if((err = readHeader()) == CEC_ERR_OK_F0)
    err = CEC_ERR_OK;
    
  return err;
}

/*------------------ECRITURE------------------*/

/*******************************************************
 * Function: moveStart()
 *******************************************************/
int Moteur::moveStart(int temps_ms)
{
  int err = 0;
  byte *pt;
  
  #ifndef CODE_LIGHT
  if(bEchoOn)
    SerialOut.print(F("moveStart: "));
  #endif  
  if (temps_ms > 0)
  {
    #ifndef CODE_LIGHT
    if(bEchoOn)
    {
      SerialOut.print(temps_ms, DEC);
      SerialOut.print(F("ms.."));
    }
    #endif
    delay(temps_ms); //start programmé
  }
  
  // data
  pt = trameMove;
  *pt++ = (byte) ACT_START_IMM; //marche avant vitesse paramétrée
  for(int i = 1; i  < CEC_LENGTH_MOVE; i++)
    *pt++ = (byte)0;
  //heure (1 octet) + minute (1 octet) + seconde (1 octet) + ms (2octets)
  //reserve (2octets) + reserve (2octets) + reserve (2octets)
  
  if((err = sendTrame(CEC_CMD_MOVE, trameMove, CEC_LENGTH_MOVE, CEC_WITHOUT_CHECKSUM)) < 0)
  {
    #ifndef CODE_LIGHT
    if(bEchoOn)
    {
      SerialOut.print(F("Err."));
      SerialOut.println(err, DEC);
    }
    #endif
  }
  else
  {
    if((err = readHeader()) == CEC_ERR_OK_F0)
      err = CEC_ERR_OK;
    #ifndef CODE_LIGHT
    if(bEchoOn)
    {
      if(err == CEC_ERR_OK)
        SerialOut.println(F("OK"));
      else
      {
        SerialOut.print(F("KO:"));
        SerialOut.println(err);
      }
    }
    #endif
  }
    
  return err;
}

/*******************************************************
 * Method: moveStop()
 ******************************************************/
int Moteur::moveStop(int temps_ms)
{
  int err = 0;
  byte *pt;
  
  #ifndef CODE_LIGHT
  if(bEchoOn)
    SerialOut.print(F("moveStop: "));
  #endif
  if (temps_ms > 0)
  {
    #ifndef CODE_LIGHT
    if(bEchoOn)
    {
      SerialOut.println(temps_ms, DEC);
      SerialOut.print(F("ms.."));
    }
    #endif
    delay(temps_ms); //start programmé
  }

  // data
  pt = trameMove;
  *pt++ = (byte) ACT_STOP_IMM; //marche avant vitesse paramétrée
  for(int i = 1; i  < CEC_LENGTH_MOVE; i++)
    *pt++ = (byte)0;
  //heure (1 octet) + minute (1 octet) + seconde (1 octet) + ms (2octets)
  //reserve (2octets) + reserve (2octets) + reserve (2octets)
  
  if((err = sendTrame(CEC_CMD_MOVE, trameMove, CEC_LENGTH_MOVE, CEC_WITHOUT_CHECKSUM)) < 0)
  {
    #ifndef CODE_LIGHT
    if(bEchoOn)
    {
      SerialOut.print(F("Err."));
      SerialOut.println(err, DEC);
    }
    #endif
  }
  else
  {
    if((err = readHeader()) == CEC_ERR_OK_F0)
      err = CEC_ERR_OK;
    #ifndef CODE_LIGHT
    if(bEchoOn)
    {
      if(err == CEC_ERR_OK)
        SerialOut.println(F("OK"));
      else
      {
        SerialOut.print(F("KO:"));
        SerialOut.println(err);
      }
    }
    #endif
  }
    
  return err;
}

/*******************************************************
 * Method: movePause()
 ******************************************************/
int Moteur::movePause(int temps_ms)
{
  int err = 0;
  byte *pt;

  #ifndef CODE_LIGHT
  if(bEchoOn)
    SerialOut.print(F("\nmovePause: "));
  #endif
  if (temps_ms > 0)
  {
    pt = trameMove;
    *pt++ = (byte) ACT_STOP_IMM; // Stop immédiat
    for(int i = 1; i  < CEC_LENGTH_MOVE; i++)
      *pt++ = (byte)0;
    //heure (1 octet) + minute (1 octet) + seconde (1 octet) + ms (2octets)
    //reserve (2octets) + reserve (2octets) + reserve (2octets)

    err = sendTrame(CEC_CMD_MOVE, trameMove, CEC_LENGTH_MOVE, CEC_WITHOUT_CHECKSUM);

    #ifndef CODE_LIGHT
    if(bEchoOn)
    {
      SerialOut.println(temps_ms, DEC);
      SerialOut.print(F("ms.."));
    }
    #endif
    delay(temps_ms);

        SerialOut.print(F(" Reprise: "));

    pt = trameMove;
    *pt++ = (byte) ACT_START_IMM; // Start immédiat  
    for(int i = 1; i  < CEC_LENGTH_MOVE; i++)
      *pt++ = (byte)0;
    //heure (1 octet) + minute (1 octet) + seconde (1 octet) + ms (2octets)
    //reserve (2octets) + reserve (2octets) + reserve (2octets)

    if(err = sendTrame(CEC_CMD_MOVE, trameMove, CEC_LENGTH_MOVE, CEC_WITHOUT_CHECKSUM) < 0)
    {
      #ifndef CODE_LIGHT
      if(bEchoOn)
      {
        SerialOut.print(F("Err."));
        SerialOut.println(err, DEC);
      }
      #endif
    }
    else
    {
    if((err = readHeader()) == CEC_ERR_OK_F0)
      err = CEC_ERR_OK;
    #ifndef CODE_LIGHT
    if(bEchoOn)
    {
      if(err == CEC_ERR_OK)
        SerialOut.println(F("OK"));
      else
      {
        SerialOut.print(F("KO:"));
        SerialOut.println(err);
      }
    }
    #endif
    }
      
    return err;
  }
}


/*******************************************************
 * Method: moveSpeedLimit()
 ******************************************************/
int Moteur::moveSpeedLimit(float Speed)
{
  int err = 0;
  int v = 0;
  byte *pt;
  byte MSByte, LSByte;

  #ifndef CODE_LIGHT
  if(bEchoOn)
    SerialOut.print(F("moveSpeedLimit("));
    SerialOut.print(Speed, DEC);
    SerialOut.print(F("m/s): "));
  #endif
  v = ((Speed*100.0f) > MAX_SPEED_CM_S) ? MAX_SPEED_CM_S :
      (Speed <= 0.0f) ? 0 :
      (Speed*100.0f);
  MSByte = (byte) ((v >> 8) & 0x00FF); // Vitesse limite poids fort
  LSByte = (byte) (v & 0x00FF); // Vitesse limite poids faible
  
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(v, DEC);
    SerialOut.print(F("cm/s("));
    SerialOut.print((MSByte << 8)+LSByte, DEC);
    SerialOut.print(F(").."));
  }
  #endif

  pt = trameMove;
  *pt++ = (byte) ACT_VITESSE_LIMITE; //marche avant vitesse limite paramétrée
  *pt++ = MSByte; // Vitesse limite poids fort
  *pt++ = LSByte; // Vitesse limite poids faible
  for(int i = 3; i  < CEC_LENGTH_MOVE; i++)
    *pt++ = (byte)0;
    
  if(err = sendTrame(CEC_CMD_MOVE, trameMove, CEC_LENGTH_MOVE, CEC_WITHOUT_CHECKSUM) < 0)
  {
    #ifndef CODE_LIGHT
    if(bEchoOn)
    {
      SerialOut.print(F("Err."));
      SerialOut.println(err, DEC);
    }
    #endif
  }
  else
  {
    if((err = readHeader()) == CEC_ERR_OK_F0)
      err = CEC_ERR_OK;
    #ifndef CODE_LIGHT
    if(bEchoOn)
    {
      if(err == CEC_ERR_OK)
        SerialOut.println(F("OK"));
      else
      {
        SerialOut.print(F("KO:"));
        SerialOut.println(err);
      }
    }
    #endif
   }
  return err;
}

/*******************************************************
 * Method: setConfigLongueurPiste()
 ******************************************************/
void Moteur::setConfigLongueurPiste(float LongueurPiste)
{
  valUInt=round(LongueurPiste*100.0f);
  // conversion vers variables internes
  convertUIntToHex(&trameConfig.longueurPiste.MSB,
                   &trameConfig.longueurPiste.LSB, valUInt);
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Long. piste Course="));
    SerialOut.print(valUInt, DEC);
    SerialOut.println(F("cm"));
  }
  #endif
}

/*******************************************************
 * Method: setConfigMotorisationType()
 ******************************************************/
void Moteur::setConfigMotorisationType(int Choix)
{
  SerialOut.print(F("\nMotorisation: "));
  if (Choix==CEC_TYPE_TRACTION)
  {
    trameConfig.configVehicule = CEC_TYPE_TRACTION;
    SerialOut.println(F("Traction"));
  }
  else
  {
    trameConfig.configVehicule = CEC_TYPE_PROPULSION;
    SerialOut.println(F("Propultion"));
  }
}

/*******************************************************
 * Method: setConfigTimeCourse()
 * fonction permettant de set le temps max de la course
 ******************************************************/
void Moteur::setConfigTimeCourse(float Temps)
{
  if (Temps <= 60.0f)
    valUInt = round(Temps * 1000.0f);
  else
    valUInt = 60000;

  // conversion vers variables internes
  convertUIntToHex(&trameConfig.tempsMaxCourse.MSB,
                   &trameConfig.tempsMaxCourse.LSB, valUInt);
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Temps Course max="));
    SerialOut.print(valUInt, DEC);
    SerialOut.println(F("ms"));
  }
  #endif
}

/*******************************************************
 * Method: setConfigCourseEndType()
 ******************************************************/
void Moteur::setConfigCourseEndType(int EndType)
{
  if(EndType == CEC_FIN_DE_COURSE_LIGNE) 
  {
    trameConfig.configCourse = CEC_FIN_DE_COURSE_LIGNE;
    SerialOut.println(F("Fin course dét. ligne"));
  }
  else // (EndType == CEC_FIN_DE_COURSE_ZONE)
  {
    trameConfig.configCourse = CEC_FIN_DE_COURSE_ZONE;
    SerialOut.println(F("Fin course zone"));
  }
}

/*******************************************************
 * Method: setConfigDiametreRoues()
 ******************************************************/
void Moteur::setConfigDiametreRoues(float DiamRoues)
{

  valUInt=round(DiamRoues*10.0f);
  convertUIntToHex(&trameConfig.DiametreRoue.MSB,
                   &trameConfig.DiametreRoue.LSB, valUInt);
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Diam. Roues:"));
    SerialOut.print(valUInt, DEC);
    SerialOut.println(F("mm"));
  }
  #endif
}

/*******************************************************
 * Method: setConfigSpeed()
 * fonction permettant de régler les zones
 ******************************************************/
void Moteur::setConfigSpeed(float Speed, float Temps, int zone)
{
  ///Partie Zone///
  if ((valInt = zone) > 9)
    valInt = 9;
    
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("\nsetConfigSpeed["));
    SerialOut.print(valInt,DEC);
    SerialOut.print(F("]: "));
  }
  #endif
  ///Partie Vitesse///
  //conversion de la valeur de vitesse en entier (2 chiffres après la virgule)//
  valUInt = round((float)Speed * 100.0f);
  if(valUInt > MAX_SPEED_CM_S)
    valUInt = 1200;
  
  convertUIntToHex(&trameConfig.segment[valInt].vitesse.MSB,
                   &trameConfig.segment[valInt].vitesse.LSB, valUInt);
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("speed="));
    SerialOut.print(valUInt, DEC);
    SerialOut.print(F("cm/s, "));
  }
  #endif
  
  ///Partie Temps///
  unsigned int tempsMax;
  convertHexToUInt(trameConfig.tempsMaxCourse.MSB,
                   trameConfig.tempsMaxCourse.LSB, &tempsMax);
  valUInt = round(Temps * 1000.0f);
  
  if (valUInt > tempsMax)
  {
    trameConfig.segment[valInt].temps.MSB = trameConfig.tempsMaxCourse.MSB;
    trameConfig.segment[valInt].temps.LSB = trameConfig.tempsMaxCourse.LSB;
  }
  else
    convertUIntToHex(&trameConfig.segment[valInt].temps.MSB,
                     &trameConfig.segment[valInt].temps.LSB, valUInt);
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Time="));
    SerialOut.print(valUInt, DEC);
    SerialOut.print(F("ms"));
  }
  #endif
}

/*******************************************************
 * Method: configMoveForward()
 *******************************************************/
int Moteur::configMoveForward(unsigned int temps_ms, unsigned int Speed)
{
  int err = 0;
  byte *pt;
  
  valUInt = (Speed > MAX_SPEED_CM_S) ? MAX_SPEED_CM_S : 
            (Speed < 0) ? 0 : Speed;
    
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("moveForward at "));
    SerialOut.print(valUInt,DEC);
    SerialOut.print(F("cm/s "));
  }
  #endif
  // Defini la vitesse de consigne dans la config zone 0
  convertUIntToHex(&trameConfig.segment[0].vitesse.MSB,
                   &trameConfig.segment[0].vitesse.LSB, valUInt);
  // Envoi de la trame de config
  if((err = sendConfig()) < 0)
    return err;
    
  tempsDebut = millis();
  tempsEcoule = tempsDebut;
  while ((tempsEcoule - tempsDebut) < temps_ms)
  {
//    motorSerial.write((byte) CEC_CMD_MOVE); //commande ACTION
    pt = trameMove;
    *pt++ = (byte) ACT_START_IMM; //marche avant vitesse paramétrée
    for(int i = 1; i  < CEC_LENGTH_MOVE; i++)
      *pt++ = (byte)0;
  //heure (1 octet) + minute (1 octet) + seconde (1 octet) + ms (2octets)
  //reserve (2octets) + reserve (2octets) + reserve (2octets)

    if(err = sendTrame(CEC_CMD_MOVE, trameMove, CEC_LENGTH_MOVE, CEC_WITHOUT_CHECKSUM))
    {
      #ifndef CODE_LIGHT
      if(bEchoOn)
      {
        SerialOut.print(F("err."));
        SerialOut.print(err,DEC);
      }
      #endif
      return err;
    }
    tempsEcoule = millis(); //stockage de la nouvelle heure
  }
  SerialOut.print(F("OK"));
  return err;
}

/*******************************************************
 * Method: configMoveBackward()
 *******************************************************/
int Moteur::configMoveBackward(unsigned int temps_ms, unsigned int Speed)
{
  int err = 0;
  byte *pt;
  
  valUInt = (Speed > MAX_SPEED_CM_S) ? MAX_SPEED_CM_S : 
           (Speed < 0) ? 0 : Speed;
               
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("moveBackward at "));
    SerialOut.print(valUInt,DEC);
    SerialOut.print(F("cm/s "));
  }
  #endif
  // Defini la vitesse de consigne dans la config zone 0
  convertUIntToHex(&trameConfig.segment[0].vitesse.MSB,
                   &trameConfig.segment[0].vitesse.LSB, valUInt);
  if((err = sendConfig()) < 0)
    return err;
      
  tempsDebut = millis();
  tempsEcoule = tempsDebut;
  while ((tempsEcoule - tempsDebut) < temps_ms)
  {
//    motorSerial.write((byte) CEC_CMD_MOVE); //commande ACTION
    pt = trameMove;
    *pt++ = (byte) ACT_MAR_LENTE; //marche arrière vitesse paramétrée
    for(int i = 1; i  < CEC_LENGTH_MOVE; i++)
      *pt++ = (byte)0;
  //heure (1 octet) + minute (1 octet) + seconde (1 octet) + ms (2octets)
  //reserve (2octets) + reserve (2octets) + reserve (2octets)

    if(err = sendTrame(CEC_CMD_MOVE, trameMove, CEC_LENGTH_MOVE, CEC_WITHOUT_CHECKSUM))
    {
      #ifndef CODE_LIGHT
      if(bEchoOn)
      {
        SerialOut.print(F("err."));
        SerialOut.print(err,DEC);
      }
      #endif
      return err;
    }
    tempsEcoule = millis(); //stockage de la nouvelle heure
  }
  SerialOut.print(F("OK"));
  return err;
}


/*------------------LECTURE------------------*/

/*******************************************************
 * Method: getMesureNum()
 * Récupère le numéro des mesures de la trame mesures
 *******************************************************/
unsigned int Moteur::getMesureNum()
{
  convertHexToUInt(trameMesures.numMesure.MSB, 
                   trameMesures.numMesure.LSB, &valUInt);

  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Num. mesure: "));
    SerialOut.println(valUInt, DEC);
  }
  #endif
    
  return valUInt;
}


/*******************************************************
 * Method: getMesureInst()
 * Récupère l'instant des mesures de la trame mesures
 *******************************************************/
unsigned int Moteur::getMesureInst()
{
  convertHexToUInt(trameMesures.instantMesure.MSB, 
                   trameMesures.instantMesure.LSB, &valUInt);

  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Instant mesure 1: "));
    SerialOut.print(valUInt, DEC);
    SerialOut.println(F("ms"));
  }
  #endif
    
  return valUInt;
}

/*******************************************************
 * Method: getMesureInstant()
 * Récupère l'instant des mesures de la trame mesures
 *******************************************************/
unsigned int Moteur::getMesureInstant()
{
  convertHexToUInt(trameMesures.mes.instant.MSB, 
                   trameMesures.mes.instant.LSB, &valUInt);

  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Instant mesure 2: "));
    SerialOut.print(valUInt, DEC);
    SerialOut.println(F("ms"));
  }
  #endif
    
  return valUInt;
}

/*******************************************************
 * Method: getMesureAccel()
 * Récupère les données de l'accéléromètre de la trame mesures
 *******************************************************/
int Moteur::getMesureAccel(int axe)
{
  #ifndef CODE_LIGHT
  if(bEchoOn)
    SerialOut.print(F("Val. accéléro. axe "));
  #endif
  switch(axe)
  {
    case CEC_ACCEL_AXE_X:
      convertHexToInt(trameMesures.mes.accel.car[0], 
                      trameMesures.mes.accel.car[1], &valInt);
      #ifndef CODE_LIGHT
      if(bEchoOn)
        SerialOut.print(F("X:"));
        SerialOut.print(valInt, DEC);
        SerialOut.println(F("mG"));
      #endif
      break;
      
    case CEC_ACCEL_AXE_Y:
      convertHexToInt(trameMesures.mes.accel.car[2], 
                      trameMesures.mes.accel.car[3], &valInt);
      #ifndef CODE_LIGHT
      if(bEchoOn)
      {
        SerialOut.print(F("Y:"));
        SerialOut.print(valInt, DEC);
        SerialOut.println(F("mG"));
      }
      #endif
      break;

    case CEC_ACCEL_AXE_Z:
      convertHexToInt(trameMesures.mes.accel.car[4], 
                      trameMesures.mes.accel.car[5], &valInt);
      #ifndef CODE_LIGHT
      if(bEchoOn)
      {
        SerialOut.print(F("Z:"));
        SerialOut.print(valInt, DEC);
        SerialOut.println(F("mG"));
      }
      #endif
      break;

    default:
      #ifndef CODE_LIGHT
      if(bEchoOn)
        SerialOut.println(F("err."));
      #endif
      break;
  }
  return valInt;
}

/*******************************************************
 * Method: getMesureGyro()
 * Récupère les données du gyromètre de la trame mesures
 *******************************************************/
int Moteur::getMesureGyro(int axe)
{
  #ifndef CODE_LIGHT
  if(bEchoOn)
    SerialOut.print(F("Val. gyro. axe "));
  #endif    
  switch(axe)     
  {
    case 1:
      convertHexToInt(trameMesures.mes.gyro.car[0], 
                      trameMesures.mes.gyro.car[1], &valInt);
      #ifndef CODE_LIGHT
      if(bEchoOn)
      {
        SerialOut.print(F("X:"));
        SerialOut.print(valInt, DEC);
        SerialOut.println(F("mRad/s"));
      }
      #endif
      break;
      
    case 2:
      convertHexToInt(trameMesures.mes.gyro.car[2], 
                      trameMesures.mes.gyro.car[3], &valInt);
      #ifndef CODE_LIGHT
      if(bEchoOn)
      {
        SerialOut.print(F("Y:"));
        SerialOut.print(valInt, DEC);
        SerialOut.println(F("mRad/s"));
      }
      #endif
      break;
      
    case 3:
      convertHexToInt(trameMesures.mes.gyro.car[4], 
                      trameMesures.mes.gyro.car[5], &valInt);
      #ifndef CODE_LIGHT
      if(bEchoOn)
      {
        SerialOut.print(F("Z:"));
        SerialOut.print(valInt, DEC);
        SerialOut.println(F("mRad/s"));
      }
      #endif
      break;

    default:
      #ifndef CODE_LIGHT
      if(bEchoOn)
        SerialOut.println(F("err."));
      #endif
      break;
  }
  return valInt;
}

/*******************************************************
 * Method: getMesureMagneto()
 * Récupère les données du magnétomètre de la trame mesures
 *******************************************************/
int Moteur::getMesureMagneto(int axe)
{
  #ifndef CODE_LIGHT
  if(bEchoOn)
    SerialOut.print(F("Val. magnéto. axe "));
  #endif
  
  switch(axe)
  {
    case 1:
      convertHexToInt(trameMesures.mes.magneto.car[0], 
                      trameMesures.mes.magneto.car[1], &valInt);
      #ifndef CODE_LIGHT
      if(bEchoOn)
      {
        SerialOut.print(F("X:"));
        SerialOut.print(valInt, DEC);
        SerialOut.println(F("mT"));
      }
      #endif
      break;
      
    case 2:
      convertHexToInt(trameMesures.mes.magneto.car[2], 
                      trameMesures.mes.magneto.car[3], &valInt);
      #ifndef CODE_LIGHT
      if(bEchoOn)
      {
        SerialOut.print(F("Y:"));
        SerialOut.print(valInt, DEC);
        SerialOut.println(F("mT"));
      }
      #endif
      break;

    default:
      #ifndef CODE_LIGHT
      if(bEchoOn)
        SerialOut.println(F("err."));
      #endif
      break;
  }
  return valInt;
}

/*******************************************************
 * Method: getMesureRpm()
 * Récupère la fréquence de rotation pignon moteur de la
 * trame mesures
 *******************************************************/
unsigned int Moteur::getMesureRpm()
{
  convertHexToUInt(trameMesures.mes.rpmMoteur.MSB, 
                   trameMesures.mes.rpmMoteur.LSB, &valUInt);
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Val. fréq. rot. moteur: "));
    SerialOut.print(valUInt, DEC);
    SerialOut.println(F("tr/min"));
  }
  #endif
  
  return valUInt;
}

/*******************************************************
 * Method: getMesureSpeed()
 * Récupère la vitesse moteur à la roue de la trame mesures
 *******************************************************/
unsigned int Moteur::getMesureSpeed()
{
  convertHexToUInt(trameMesures.mes.vitesse.MSB, 
                   trameMesures.mes.vitesse.LSB, &valUInt);
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Val. Vit. moteur: "));
    SerialOut.print(valUInt, DEC);
    SerialOut.println(F("cm/s"));
  }
  #endif
  return valUInt;
}

/*******************************************************
 * Method: getMesureDistance()
 * Récupère la distance parcourue de la trame mesures
 *******************************************************/
unsigned int Moteur::getMesureDistance()
{
  convertHexToUInt(trameMesures.mes.distance.MSB, 
                   trameMesures.mes.distance.LSB, &valUInt);
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Val. dist. parcourue:"));
    SerialOut.print(valUInt, DEC);
    SerialOut.println(F("cm"));
  }
  #endif
  return valUInt;
}

/*******************************************************
 * Method: getMesureColor()
 * Récupère la couleur RGB détectée de la trame mesures
 *******************************************************/
unsigned int Moteur::getMesureColor()
{
  valUInt = trameMesures.mes.coulLigne;
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Val. capt. coul. ligne RGB:"));
    SerialOut.println(valUInt, DEC);
  }
  #endif
  return valUInt;
}

/*******************************************************
 * Method: getMesureTension()
 * Récupère la tension batterie de la trame mesures
 *******************************************************/
unsigned int Moteur::getMesureTension()
{
  convertHexToUInt(trameMesures.mes.tensBat.MSB, 
                   trameMesures.mes.tensBat.LSB, &valUInt);
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Val. tension moteur:"));
    SerialOut.print(valUInt, DEC);
    SerialOut.println(F("mV"));
  }
  #endif
  return valUInt;
}

/*******************************************************
 * Method: getMesureCourant()
 * Récupère le courant moteur de la trame mesures
 *******************************************************/
unsigned int Moteur::getMesureCourant()
{
  valInt = 0;
  byte octet0, octet1;
  convertHexToUInt(trameMesures.mes.courCons.MSB, 
                   trameMesures.mes.courCons.LSB, &valUInt);
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Val. intensité moteur:"));
    SerialOut.print(valUInt, DEC);
    SerialOut.println(F("mA"));
  }
  #endif
  return valUInt;
}

/*******************************************************
 * Method: getMesureTemp()
 * Récupère la température moteur de la trame mesures
 *******************************************************/
unsigned int Moteur::getMesureTemp()
{
  convertHexToUInt(trameMesures.mes.temperature.MSB, 
                   trameMesures.mes.temperature.LSB, &valUInt);
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Temp. moteur:"));
    SerialOut.print(valUInt / 10, DEC);
    SerialOut.println(F("°C"));
  }  
  #endif
  return valUInt;
}

/*******************************************************
 * Method: getMesureCourseEtat()
 * Récupère l'etat de course de la trame mesures
 *******************************************************/
unsigned int Moteur::getMesureCourseEtat()
{
  valUInt = trameMesures.mes.etatCourse.etat;
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Course Etat:"));
    SerialOut.println(valInt, DEC);
  }
  #endif
  return valInt;
}

/*******************************************************
 * Method: getMesureCourseSegment()
 * Récupère le segment de course de la trame mesures
 *******************************************************/
unsigned int Moteur::getMesureCourseSegment()
{
  valUInt = trameMesures.mes.etatCourse.segment + 1;
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Course Segment:"));
    SerialOut.println(valUInt, DEC);
  }
  #endif
  return valUInt;
}

/*******************************************************
 * Method: getInfoConnex()
 * Récupère l'état de connexion moteur de la trame mesures
 *******************************************************/
int Moteur::getInfoConnex()
{
  #ifndef CODE_LIGHT
  if(bEchoOn)
    SerialOut.print(F("Connex. Moteur:"));
  #endif
  if ((trameInfo.etatConnexCarte & DETECT_CARTE_MOTEUR) == 0) {
    if(bEchoOn)
      SerialOut.println(F("Err. interne moteur!"));
    return CEC_ERR_INTERNAL;
  }
  if(bEchoOn)
    SerialOut.println(F(" Ok"));
  return CEC_ERR_OK;
}

/*******************************************************
 * Method: getConfigLongueurPiste()
 * Récupère la longueur de la trame Config
 *******************************************************/
unsigned int Moteur::getConfigLongueurPiste()
{
  convertHexToInt(trameConfig.longueurPiste.MSB,
                  trameConfig.longueurPiste.LSB, &valInt);
  valFloat = (float)valInt / 100.0;
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Long. piste:"));
    SerialOut.print(valFloat);
    SerialOut.println(F("m"));
  }
  #endif
  return valInt;
}

/*******************************************************
 * Method: getConfigTailleDamier()
 * Récupère la taille du damier de la trame Config
 *******************************************************/
unsigned int Moteur::getConfigTailleDamier()
{
  valUInt =  trameConfig.tailleDamier;
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Taille damier:"));
    SerialOut.print(valUInt);
    SerialOut.println(F("cm"));
  }
  #endif
  return valUInt;
}

/*******************************************************
 * Method: getConfigDiametreRoues()
 * Récupère le diamètre des roues de la trame Config
 *******************************************************/
unsigned int Moteur::getConfigDiametreRoues()
{
  convertHexToUInt(trameConfig.DiametreRoue.MSB,
                   trameConfig.DiametreRoue.LSB, &valUInt);
  valFloat = (float)valUInt / 10.0;
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Diam. roues:"));
    SerialOut.print(valFloat);
    SerialOut.println(F("mm"));
  }
  #endif
  return valUInt;
}

/*******************************************************
 * Method: getConfigPignon()
 * Récupère le nombre de dents pignon de la trame Config
 *******************************************************/
unsigned int Moteur::getConfigPignon()
{
  valUInt = trameConfig.nDentsPignon;
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Nb. dents pignon:"));
    SerialOut.print(valUInt);
  }
  #endif
  return valUInt;
}

/*******************************************************
 * Method: getConfigCouronne()
 * Récupère le nombre de dents couronne de la trame Config
 *******************************************************/
unsigned int Moteur::getConfigCouronne()
{
  valUInt = trameConfig.nDentsCouronne;
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Nb. dents couronne:"));
    SerialOut.print(valUInt);
  }
  #endif
  return valUInt;
}


/*******************************************************
 * Method: getConfigVehicule()
 * Récupère le type de motorisation de la trame Config
 *******************************************************/
unsigned int Moteur::getConfigVehicule()
{
  valUInt = trameConfig.configVehicule & CEC_TYPE_TRACTION;
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Type motorisation:"));
    if(valUInt == 1)
      SerialOut.println(F(" Tract."));
    else
      SerialOut.println(F(" Prop."));
  }
  #endif
  return valUInt;
}

/*******************************************************
 * Method: getConfigCourse()
 * Récupère le type de fin de course de la trame Config
 *******************************************************/
unsigned int Moteur::getConfigCourse()
{
  valUInt = trameConfig.configCourse & (CEC_TYPE_ARRET_LIGNE | CEC_TYPE_DAMIER);
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Type course:Arret "));
    if(valUInt & CEC_TYPE_ARRET_LIGNE)
      SerialOut.print(F("ligne"));
    else
      SerialOut.println(F("zone"));
  
    if(valUInt & CEC_TYPE_DAMIER)
      SerialOut.print(F("Avec"));
    else
      SerialOut.print(F("Sans"));
    SerialOut.println(F(" damier"));
  }    
  #endif
  return valUInt;
}

/*******************************************************
 * Method: getConfigPeriodeEch()
 * Récupère la periode d'echantillonnage des mesures
 * de la trame Config
 *******************************************************/
unsigned int Moteur::getConfigPeriodeEch()
{
  valUInt = trameConfig.periodeEch;
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Période échantil.:"));
    SerialOut.print(valUInt);
    SerialOut.println(F("ms"));
  }
  #endif
  return valUInt;
}

//// lecture des couleurs de lignes de départ, inter1, inter2 et fin trameConfig[11, 12, 13, 14]: Pas encore utilisé

/*******************************************************
 * Method: getConfigTempsMaxCourse()
 * Récupère le temps maximum de la course de la trame Config
 *******************************************************/
unsigned int Moteur::getConfigTempsMaxCourse()
{
  convertHexToUInt(trameConfig.tempsMaxCourse.MSB,
                   trameConfig.tempsMaxCourse.LSB, &valUInt);
  valFloat = (float)valUInt / 1000.0;
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Temps course:"));
    SerialOut.print(valUInt);
    SerialOut.println(F("ms"));
  }
  #endif
  return valUInt;
}

/*******************************************************
 * Method: getConfigVitesseMin()
 * Récupère la vitesse minimale de la trame Config
 *******************************************************/
unsigned int Moteur::getConfigVitesseMin()
{
  convertHexToUInt(trameConfig.vitesseMotMin.MSB,
                   trameConfig.vitesseMotMin.LSB, &valUInt);
  valFloat = (float)valUInt / 100.0;
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Vit. min:"));
    SerialOut.print(valFloat);
    SerialOut.println(F("m/s"));
  }
  #endif
  return valUInt;
}

/*******************************************************
 * Method: getConfigVitesseMax()
 * Récupère la vitesse maximale de la course de la trame 
 * Config
 *******************************************************/
unsigned int Moteur::getConfigVitesseMax()
{
  convertHexToUInt(trameConfig.vitesseMotMax.MSB,
                   trameConfig.vitesseMotMax.LSB, &valUInt);
  valFloat = (float)valUInt / 100.0;
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Vit. max:"));
    SerialOut.print(valFloat);
    SerialOut.println(F("m/s"));
  }
  #endif
  return valUInt;
}

/*******************************************************
 * Method: getConfigSegmentVitesse()
 * Récupère la vitesse finale du segment indiqué de la 
 * trame Config
 *******************************************************/
unsigned int Moteur::getConfigSegmentVitesse(int numSegment)
{
  #ifndef CODE_LIGHT
  if(bEchoOn)
    SerialOut.print(F("Vit. segment"));
  #endif
      
  if(numSegment < 0 || numSegment >= CEC_CONFIG_SEGMENTS) 
  {
    #ifndef CODE_LIGHT
    if(bEchoOn)
      SerialOut.print(F(": numSegment err."));
    #endif
    
    return CEC_ERR;
  }
    
  convertHexToUInt(trameConfig.segment[numSegment].vitesse.MSB,
                   trameConfig.segment[numSegment].vitesse.LSB, &valUInt);
  valFloat = valUInt;
  valFloat /= 100.0;
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(numSegment);
    SerialOut.print(F(":"));
    SerialOut.print(valFloat);
    SerialOut.println(F("m/s"));
  }
  #endif
  
  return valUInt;
}

/*******************************************************
 * Method: getConfigSegmentTemps()
 * Récupère le Temps finale du segment indiqué de la 
 * trame Config
 *******************************************************/
unsigned int Moteur::getConfigSegmentTemps(int numSegment)
{
  #ifndef CODE_LIGHT
  if(bEchoOn)
    SerialOut.print(F("Temps segment"));
  #endif
      
  if(numSegment < 0 || numSegment >= CEC_CONFIG_SEGMENTS) 
  {
    #ifndef CODE_LIGHT
    if(bEchoOn)
      SerialOut.print(F(": numSegment err."));
    #endif
    
    return CEC_ERR_SEG;
  }    
  convertHexToUInt(trameConfig.segment[numSegment].temps.MSB,
                   trameConfig.segment[numSegment].temps.LSB, &valUInt);
  valFloat = valUInt;
  valFloat /= 1000.0;
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(numSegment);
    SerialOut.print(F(":"));
    SerialOut.print(valFloat);
    SerialOut.println(F("s"));
  }
  #endif
  
  return valUInt;
}

/*******************************************************
 * Method: getConfigVitesseLente()
 * Récupère la vitesse lente de la trame Config
 *******************************************************/
unsigned int Moteur::getConfigVitesseLente()
{
  #ifndef CODE_LIGHT
  if(bEchoOn)
    SerialOut.print(F("Vit. lente:"));
  #endif
      
  convertHexToUInt(trameConfig.vitesseMotMax.MSB,
                   trameConfig.vitesseMotMax.LSB, &valUInt);
  valFloat = (float)valUInt / 1000.0;
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(valFloat);
    SerialOut.println(F("m/s"));
  }
  #endif
  return valUInt;
}

/*******************************************************
 * Method: getConfigTimeHeure()
 * Récupère les heures du temps actuel de la trame Config
 *******************************************************/
unsigned int Moteur::getConfigTimeHeures()
{

 valUInt = trameConfig.tempsMot.heure;
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Temps heure:"));
    SerialOut.print(valUInt, DEC);
  }
  #endif
  return valUInt;
}

/*******************************************************
 * Method: getConfigTimeMinutes()
 * Récupère les minutes du temps actuel de la trame Config
 *******************************************************/
unsigned int Moteur::getConfigTimeMinutes()
{
  valUInt = trameConfig.tempsMot.minute;
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Temps min.:"));
    SerialOut.println(valUInt, DEC);
  }
  #endif
  return valUInt;
}

/*******************************************************
 * Method: getConfigTimeSecondes()
 * Récupère les secondes du temps actuel de la trame Config
 *******************************************************/
unsigned int Moteur::getConfigTimeSecondes()
{
  valUInt = trameConfig.tempsMot.seconde;
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Temps Sec."));
    SerialOut.println(valUInt, DEC);
  }
  #endif
  return valUInt;
}

/*******************************************************
 * Method: getConfigTimeMillisecondes()
 * Récupère les millisecondes du temps actuel de la trame Config
 *******************************************************/
unsigned int Moteur::getConfigTimeMillisecondes()
{
  convertHexToUInt(trameConfig.tempsMot.milliseconde.MSB,
                   trameConfig.tempsMot.milliseconde.LSB, &valUInt);
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Temps MilliS.: "));
    SerialOut.print(valUInt);
    SerialOut.println(F("ms"));
  }
  #endif
  return valUInt;
}

  // Fonctions liées au bilan
/*******************************************************
 * Method: getBilanFinalTime()
 * fonction permettant l'affichage de la valeur bilan de
 * temps
 *******************************************************/
unsigned int Moteur::getBilanFinalTime()
{
  convertHexToUInt(trameBilan.tempsCourseMSB, 
                   trameBilan.tempsCourseLSB, &valUInt);

  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Temps Bilan:"));
    SerialOut.print(valUInt, DEC);
    SerialOut.println(F("ms"));
  }
  #endif
  return valUInt;
}

/*******************************************************
 * Method: getBilanFinalDistance()
 * fonction permettant l'affichage de la valeur bilan de
 * Distance
 *******************************************************/
unsigned int Moteur::getBilanFinalDistance()
{
  convertHexToUInt(trameBilan.distanceParcourueRoueCmMSB,
                   trameBilan.distanceParcourueRoueCmLSB, &valUInt);
  
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Dist. finale:"));
    SerialOut.print(valUInt, DEC);
    SerialOut.println(F("cm"));
  }
  #endif
  return valUInt;
}

/*******************************************************
 * Method: getBilanFinalVmoy()
 * fonction permettant l'affichage de la valeur bilan de
 * Vitesse moyenne atteinte
 *******************************************************/
unsigned int Moteur::getBilanFinalVmoy()
{
  convertHexToUInt(trameBilan.vitesseMoyenneCm_sMSB,
                   trameBilan.vitesseMoyenneCm_sLSB, &valUInt);

  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Vit. moy.:"));
    SerialOut.print(valUInt, DEC);
    SerialOut.println(F("cm/sec"));
  }
  #endif
  return valUInt;
}

/*******************************************************
 * Method: getBilanFinalVmax()
 * fonction permettant l'affichage de la valeur bilan de
 * Vitesse maximale atteinte
 *******************************************************/
unsigned int Moteur::getBilanFinalVmax()
{
  convertHexToUInt(trameBilan.vitesseMaxiCm_sMSB,
                   trameBilan.vitesseMaxiCm_sLSB, &valUInt);

  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Vit. max.:"));
    SerialOut.print(valUInt, DEC);
    SerialOut.println(F("cm/sec"));
  }
  #endif
  return (int)valUInt;
}

/*******************************************************
 * Method: getBilanFinalCumulCourant()
 * fonction permettant l'affichage de la valeur bilan du
 * cumul de courant
 *******************************************************/
unsigned int Moteur::getBilanFinalCumulCourant()
{
  convertHexToUInt(trameBilan.consoCourantCumulMSB,
                   trameBilan.consoCourantCumulLSB, &valUInt);

  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Cumul Courant:"));
    SerialOut.print(valUInt, DEC);
    SerialOut.println(F("A/sec"));
  }
  #endif
  return valUInt;
}

/*******************************************************
 * Method: getBilanFinalPconso()
 * fonction permettant l'affichage de la valeur bilan de
 * la puissance consommée
 *******************************************************/
unsigned int Moteur::getBilanFinalPconso()
{
  convertHexToUInt(trameBilan.puissanceConsoMSB,
                   trameBilan.puissanceConsoLSB, &valUInt);

  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Puiss. cons.:"));
    SerialOut.print(valUInt, DEC);
    SerialOut.println(F("W/sec"));
  }
  #endif
  return valUInt;
}

/*******************************************************
 * Method: getBilanTypeDepart()
 *******************************************************/
unsigned int Moteur::getBilanTypeDepart()
{
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Type départ:"));
    switch(trameBilan.typeDepart)
    {
      case CEC_TYD_IMM:
        SerialOut.println(F("immédiat"));
        break;

      case CEC_TYD_PROG:
        SerialOut.println(F("programmé"));
        break;

      case CEC_TYD_NONE:
      default:
        SerialOut.println(F("indéf.")); 
        break;
    }
  }
  #endif
  if(trameBilan.typeDepart <= CEC_TYD_NONE ||
     trameBilan.typeDepart >= CEC_TYD_MAX)
    return (CEC_ERR);

    valUInt = trameBilan.typeDepart;
  return valUInt;
}

/*******************************************************
 * Method: getBilanTypeFin()
 *******************************************************/
unsigned int Moteur::getBilanTypeFin()
{  
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Type fin course:"));
    switch(trameBilan.motifFinCourse)
    {
      case CEC_TYF_FIN_ZONE:
        SerialOut.println(F("Fin zone"));
        break;

      case CEC_TYF_LIGNE_FIN:
        SerialOut.println(F("Ligne arrivée"));
        break;

      case CEC_TYF_ARRET_IMM:
        SerialOut.println(F("Arret immédiat"));
        break;

      case CEC_TYF_DIST_MAX:
        SerialOut.println(F("Dist. max."));
        break;

      case CEC_TYF_TEMPS_MAX:
        SerialOut.println(F("Temps max."));
        break;

      case CEC_TYF_NONE:
      default:
        SerialOut.println(F("Type indéf."));  // non defini
        break;
    }
  }
  #endif
  if(trameBilan.motifFinCourse <= CEC_TYF_NONE ||
     trameBilan.motifFinCourse >= CEC_TYF_MAX)
    return (CEC_ERR);

    valUInt = trameBilan.motifFinCourse;
  return valUInt;
}

/*******************************************************
 * Method: getBilanFinalPconso()
 * Récupère l'acceleration moyenne finale de la course
 *******************************************************/
int Moteur::getBilanFinalAccelMoy()
{
  convertHexToInt(trameBilan.accelerationMoyenneMSB,
                  trameBilan.accelerationMoyenneLSB, &valInt);

  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Accel. moy.:"));
    SerialOut.print(valInt, DEC);
    SerialOut.println(F("cm/sec2"));
  }
  #endif
  return valInt;
}

/*******************************************************
 * Method: getBilanAccel()
 * Récupère l'acceleration
 *******************************************************/
float Moteur::getBilanAccel()
{
  float fAccel = 0.0f;
  
  convertHexToInt(trameBilan.accelerationAcceleroMSB,
                  trameBilan.accelerationAcceleroLSB, &valInt);
  fAccel = valInt / 100.0f;
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Accel.:"));
    SerialOut.print(fAccel);
    SerialOut.println(F("m/s²"));
  }
  #endif
  return fAccel;                           
}

/*******************************************************
 * Method: getBilanVitesse()
 * Récupère la vitesse
 *******************************************************/
float Moteur::getBilanVitesse()
{
  float fSpeed = 0.0f;
  
  convertHexToInt(trameBilan.vitesseAcceleroMSB,
                  trameBilan.vitesseAcceleroLSB, &valInt);
  fSpeed = valInt/100.0f;
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Vit.:"));
    SerialOut.print(fSpeed);
    SerialOut.println(F("m/s"));
  }
  #endif
  return fSpeed;                                                     
}

/*******************************************************
 * Method: getBilanPosition()
 * Récupère la position
 *******************************************************/
float Moteur::getBilanPosition()
{
  float fPos = 0.0f;
  
  convertHexToInt(trameBilan.positionAcceleroMSB,
                  trameBilan.positionAcceleroLSB, &valInt);
  fPos = valInt / 100.0f;
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Pos.:"));
    SerialOut.print(fPos);
    SerialOut.println(F("m"));
  }
  #endif
  return fPos;            
}

/*******************************************************
 * Method: getBilanCourseEtat()
 * Récupère l'état actuel de course 
 *******************************************************/
byte Moteur::getBilanCourseEtat()
{
  byte sequence = (trameBilan.seqCourse & 0x0f);
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Etat de course:"));
    if(sequence < CEC_SEQ_COU_MAX)
      SerialOut.println(cecSeqCourseEtatTxt[sequence]);
    else 
      SerialOut.println(F("Type indéf."));  // non defini
  }
  #endif
  return sequence;
                     
}

/*******************************************************
 * Method: getBilanCourseZone()
 * Récupère la zone actuelle de course 
 *******************************************************/
byte Moteur::getBilanCourseZone()
{
  byte zone = ((trameBilan.seqCourse >> 4) & 0x0f)+1;
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.print(F("Zone de course:"));
    SerialOut.println(zone);
  }
  #endif
  return zone;                
}

/*------------------ANNEXE------------------*/

/*******************************************************
 * Method: readTrame()
 *******************************************************/
int Moteur::readTrame(byte cmd, byte *ptData, uint16_t dataSize, bool bWithCheckSum)
{
  int err = 0;
  
  // Envoie de commande de lecture de trame 
  // (no data)
  err = sendTrame(cmd, NULL, 0, bWithCheckSum);

  // lecture de trame en retour
  // Lecture Header
  if((err = readHeader()) < 0)
  {
    #ifndef CODE_LIGHT
    if(bEchoOn)
    {
      SerialOut.print(F("err. retour Header="));
      SerialOut.println(err);
      SerialOut.print(F("header: "));
      printHeader();
      SerialOut.println();
    }
    #endif
    return err;
  }
  rcvCmd = this->trameHeader.cmd;
  if(this->trameHeader.cmd != cmd)
  {
    #ifndef CODE_LIGHT
    if(bEchoOn)
    {
      SerialOut.print(F("err. retour cmd="));
      SerialOut.print(this->trameHeader.cmd);
      SerialOut.print(F("("));
      SerialOut.print(cmd);
      SerialOut.println(F(")"));
      SerialOut.print(F("header: "));
      printHeader();
      SerialOut.println();
    }
    #endif
    return CEC_ERR_WRONG_CMD;
  }

  // Lecture Data
  //this->longueurData = ((uint16_t) this->trameHeader.lengthMSB << 8) + this->trameHeader.lengthLSB;
  if(this->longueurData != dataSize + 1)
  {
    #ifndef CODE_LIGHT
    if(bEchoOn)
    {
      SerialOut.print(F("err. retour long.="));
      SerialOut.println(this->longueurData);
      SerialOut.print(F("header: "));
      printHeader();
      SerialOut.println();
    }
    #endif
    return CEC_ERR_LENGTH;
  }
  if((err = readData(dataSize, ptData)) < 0)
  {
    #ifndef CODE_LIGHT
    if(bEchoOn)
    {
      SerialOut.print(F("err. retour data="));
      SerialOut.println(err);
    }
    #endif
    return err;
  }

  //Lecture Footer
  if((err = readFooter()) < 0)
  {
    #ifndef CODE_LIGHT
    if(bEchoOn)
    {
      SerialOut.print(F("err. retour Footer="));
      SerialOut.println(err);
      SerialOut.print(F("footer: "));
      printFooter();
      SerialOut.println();
    }
    #endif
    return err;
  }
  
  // Fin sans erreur
  #ifndef CODE_LIGHT
  if(bEchoOn)
  {
    SerialOut.println(F("OK"));
  }
  #endif
  return CEC_ERR_OK;
}

/*******************************************************
 * Method: initConfig()
 * initialisation de la trame Config en mémoire
 *******************************************************/
int Moteur::initConfig()
{
  #ifndef CODE_LIGHT
  if(bEchoOn)
    SerialOut.print(F("initConfig:"));
  #endif
  memcpy(trameConfig.car, trameConfigInit, CEC_LENGTH_CONFIG);
  #ifndef CODE_LIGHT
  if(bEchoOn)
    SerialOut.println(F("OK"));
  #endif
  return CEC_ERR_OK;
}

/*******************************************************
 * Method: readConfig()
 *******************************************************/
int Moteur::readConfig()
{
  int err = CEC_ERR_OK;
  #ifndef CODE_LIGHT
  if(bEchoOn)
    SerialOut.print(F("readConfig:"));
  #endif
  
  if((err = readTrame(CEC_CMD_GET_CONFIG, trameConfig.car, CEC_LENGTH_CONFIG, CEC_WITHOUT_CHECKSUM)) >= 0)
  {
    #ifndef CODE_LIGHT
    if(bEchoOn)
      SerialOut.println(F("OK"));
    #endif
    bTrameConfigOK = true;
  }
  else
  {
    #ifndef CODE_LIGHT
    if(bEchoOn)
    {
      SerialOut.print(F("err."));
      SerialOut.println(err, DEC);
    }
    #endif
  }
  return err;
}

/*******************************************************
 * Method: ReadMesures()
 *******************************************************/
int Moteur::ReadMesures()
{
  int err = CEC_ERR_OK;
  #ifndef CODE_LIGHT
  if(bEchoOn)
    SerialOut.print(F("ReadMesures:"));
  #endif
  
  if(err = readTrame(CEC_CMD_MESURE, trameMesures.car, CEC_LENGTH_MESURE, CEC_WITHOUT_CHECKSUM) >= 0)
  {
    #ifndef CODE_LIGHT
    if(bEchoOn)
      SerialOut.println(F("OK"));
    #endif
    bTrameMesuresOK = true;
  }
  else
  {
    #ifndef CODE_LIGHT
    if(bEchoOn)
    {
      SerialOut.print(F("err."));
      SerialOut.println(err, DEC);
    }
    #endif
  }
  return err;
}

/*******************************************************
 * Method: readInfo()
 *******************************************************/
int Moteur::readInfo()
{
  int err = CEC_ERR_OK;
  #ifndef CODE_LIGHT
  if(bEchoOn)
    SerialOut.print(F("readInfo:"));
  #endif
  
  if(err = readTrame(CEC_CMD_INFO, trameInfo.car, CEC_LENGTH_INFO, CEC_WITHOUT_CHECKSUM) >= 0)
  {
    #ifndef CODE_LIGHT
    if(bEchoOn)
      SerialOut.println(F("OK"));
    #endif
    
    bTrameMesuresOK = true;
  }
  else
  {
    #ifndef CODE_LIGHT
    if(bEchoOn)
    {
      SerialOut.print(F("err."));
      SerialOut.println(err, DEC);
    }
    #endif
  }
}


/*******************************************************
 * Method: getInfoVersion()
 *******************************************************/
byte Moteur::getInfoVersion(int type)
{
  byte ver;
  
  switch(type)  
  {
  case CEC_VERSION_MERE_HARD:
    ver = trameInfo.vers_mere_hard_maj;
    break;
  case CEC_VERSION_MERE_HARD_MIN:
    ver = trameInfo.vers_mere_hard_min;
    break;
  case CEC_VERSION_MERE_SOFT:
    ver = trameInfo.vers_mere_soft_maj;
    break;
  case CEC_VERSION_MERE_SOFT_MIN:
    ver = trameInfo.vers_mere_soft_min;
    break;
  case CEC_VERSION_MOT_HARD:
    ver = trameInfo.vers_mot_hard_maj;
    break;
  case CEC_VERSION_MOT_HARD_MIN:
    ver = trameInfo.vers_mot_hard_min;
    break;
  case CEC_VERSION_MOT_SOFT:
    ver = trameInfo.vers_mot_soft_maj;
    break;
  case CEC_VERSION_MOT_SOFT_MIN:
    ver = trameInfo.vers_mot_soft_min;
    break;
  case CEC_VERSION_CAPT_HARD:
    ver = trameInfo.vers_capt_hard_maj;
    break;
  case CEC_VERSION_CAPT_HARD_MIN:
    ver = trameInfo.vers_capt_hard_min;
    break;
  case CEC_VERSION_CAPT_SOFT:
    ver = trameInfo.vers_capt_soft_maj;
    break;
  case CEC_VERSION_CAPT_SOFT_MIN:
    ver = trameInfo.vers_capt_soft_min;
    break;
  case CEC_VERSION_NONE:
  default:
    ver = CEC_ERR;
    break;
  }
  return ver;
}


/*******************************************************
 * Method: getInfoVersion()
 *******************************************************/
int Moteur::printInfoVersion(int type)
{
  byte vMaj, vMin;
  if (type > CEC_VERSION_NONE && type < CEC_VERSION_MAX)
  {
    int monType = (type - 1) | 0x01;
    vMaj = getInfoVersion(monType);
    vMin = getInfoVersion(monType + 1);
    SerialOut.print(vMaj);
    SerialOut.print(F("."));
    SerialOut.print(vMin);
    return CEC_ERR_OK;
  }
  // Erreur
  SerialOut.print(F("[Type Err.]"));
  return CEC_ERR;
}

/*******************************************************
 * Method: ReadBilan()
 *******************************************************/
int Moteur::ReadBilan()
{
  int err = CEC_ERR_OK;
#ifndef CODE_LIGHT
  if(bEchoOn)
    SerialOut.print(F("ReadBilan:"));
#endif
  
  if(err = readTrame(CEC_CMD_BILAN, trameBilan.car, CEC_LENGTH_BILAN, CEC_WITHOUT_CHECKSUM) >= 0)
  {
#ifndef CODE_LIGHT
    if(bEchoOn)
      SerialOut.println(F("OK"));
#endif
    bTrameMesuresOK = true;
  }
  else
  {
#ifndef CODE_LIGHT
    if(bEchoOn)
    {
      SerialOut.print(F("err."));
      SerialOut.println(err, DEC);
    }
#endif
  }
}

/*******************************************************
 * Function: convertIntToHex()
 *******************************************************/
void Moteur::convertIntToHex(byte *valMSB, byte *valLSB, int valeur)
{
  *valMSB = (byte)((valeur >> 8) & 0x00ff); // MSB
  *valLSB = (byte)(valeur  & 0x00ff); // LSB
}

/*******************************************************
 * Function: convertUIntToHex()
 *******************************************************/
void Moteur::convertUIntToHex(byte *valMSB, byte *valLSB, unsigned int valeur)
{
  *valMSB = (byte)((valeur >> 8) & 0x00ff); // MSB
  *valLSB = (byte)(valeur  & 0x00ff); // LSB
}

/*******************************************************
 * Function: convertHexToInt()
 *******************************************************/
void Moteur::convertHexToInt(byte valMSB, byte valLSB, int *valeur)
{
  *valeur = valMSB;
  *valeur = *valeur * 256;
  *valeur = *valeur + valLSB;
  if(*valeur > 32767)
    *valeur = *valeur - 65536;
}

/*******************************************************
 * Function: convertHexToUInt()
 *******************************************************/
void Moteur::convertHexToUInt(byte valMSB, byte valLSB, unsigned int *valeur)
{
  *valeur = valMSB;
  *valeur = *valeur * 256;
  *valeur = *valeur + valLSB;
}

/*******************************************************
 * Function: printBuffer()
 *******************************************************/
void Moteur::printBuffer(byte *buff, uint16_t dataSize)
{
  byte *pt = buff;
  for(uint16_t i = 0; i < dataSize; i++)
  {
    if(i > 0)
      SerialOut.print(F(" "));
    if(*pt < 0x10)
      SerialOut.print(F("0"));
    SerialOut.print(*pt++,HEX);      
  }
}

/*******************************************************
 * Function: printHeader()
 *******************************************************/
void Moteur::printHeader()
{
  printBuffer(this->trameHeader.car, CEC_LENGTH_HEADER);
}

/*******************************************************
 * Function: printFooter()
 *******************************************************/
void Moteur::printFooter()
{
  printBuffer(trameFooter, CEC_LENGTH_FOOTER);
}
