/******************************************************************************
 * Librairie: CEC_lib                                                         *
 *   Fichier: CEC_lib.h                                                       *
 *    Auteur: Laurent MARTIN                                                  *
 *                                                                            *
 *   Version  date        Description                                         *
 *   2.0      18/03/2022  Version opérationnelle                              *
 *   1.0      24/02/2022  Version initiale                                    *
 *                                                                            *
 ******************************************************************************/
/*******************************************************************************
 Fichier d'entête pour CEC_lib.cpp
*******************************************************************************/
#ifndef CEC_LIB_H
#define CEC_LIB_H

#include "Arduino.h"

/*******************************************************************************
 * compilation légère : enlevez la mise en commentaire de la ligne ci-dessous  *
 * pour supprimer le code des sorties texte et ainsi l'alléger en cas de       *
 * mémoire saturée :                                                           *
 *******************************************************************************/
//#define CODE_LIGHT

/*******************************************************************************
 * Pour les cartes ayant un 2ème port série materiel comme l'arduino DUE,      *                                                           
 * enlevez la mise en commentaire de la ligne ci-dessous pour l'utiliser ;     * 
 * pour l'Arduino UNO, la laisser en commentaire :                             *
 *******************************************************************************/
//#define ARDUINO_DUE

/*******************************************************************************
 * Définition des valeurs constantes                                           *
 *******************************************************************************/

// Configuration trame - checksum
#define CEC_WITHOUT_CHECKSUM 0
#define CEC_WITH_CHECKSUM    1

// commandes trame moteur
#define CEC_CMD_INFO       0x10
#define CEC_CMD_GET_CONFIG 0X20
#define CEC_CMD_CONFIG     0X21
#define CEC_CMD_MOVE       0x30
#define CEC_CMD_MESURE     0x41
#define CEC_CMD_BILAN      0x42

// Longueur trame data hors commande
#define CEC_LENGTH_HEADER  5
#define CEC_LENGTH_FOOTER  3
#define CEC_LENGTH_INFO   23
#define CEC_LENGTH_CONFIG 68
#define CEC_LENGTH_MOVE   12
#define CEC_LENGTH_MESURE 36
#define CEC_LENGTH_BILAN  23

// Valeurs de timeout
#define CEC_TIMEOUT1      100 // ms
#define CEC_TIMEOUT2      5   // ms
#define CEC_TIMEOUT3      10  // ms
#define CEC_TIMEOUT4      2  // ms
#define CEC_TIMEOUT5      10 // ms

// Actions de la trame Move
#define ACT_START_PROG      1
#define ACT_START_IMM       2
#define ACT_STOP_PROG       3
#define ACT_STOP_IMM        4
#define ACT_MAR_LENTE       5
#define ACT_MAV_LENTE       6
#define ACT_PAUSE           7
#define ACT_REPRISE         8
#define ACT_DEMARRE_FIXE    9
#define ACT_VITESSE_FIXE    10
#define ACT_VITESSE_LIMITE  11
#define ACT_MAX             12

// Sequence de course
#define CEC_SEQ_COU_RIEN           0  // 
#define CEC_SEQ_COU_ATT            1  // attente ordre course ou mouvement
#define CEC_SEQ_COU_DEM            2  // initialisation pour une course
#define CEC_SEQ_COU_CAL            3  // calcul
#define CEC_SEQ_COU_CAPT_ON        4  // activation carte capteur
#define CEC_SEQ_COU_COU            5  // dï¿½roulement de la course
#define CEC_SEQ_COU_FIN_0          6  // dernier pas de la course
#define CEC_SEQ_COU_FIN_LIGNE      7  // pas aprï¿½s passage de la ligne d'arrivï¿½e
#define CEC_SEQ_COU_FIN            8  // fin de la course
#define CEC_SEQ_COU_ANNULE         9  // annulation
#define CEC_SEQ_COU_RECUL_INIT     10 // initialisation pour le recul de la voiture
#define CEC_SEQ_COU_AVANCE_INIT    11 // initialisation pour l'avance de la voiture
#define CEC_SEQ_COU_POSITIONNE     12 // positionne la voiture
#define CEC_SEQ_COU_FIXE           13 // vitesse fixe 
#define CEC_SEQ_COU_MAX            14 // Nombre d'états de course

// Etats de Connexion Cartes;
#define DETECT_CARTE_CAPTEUR    0x01
#define DETECT_CARTE_MOTEUR     0x02

// Etats de Connexion Externe
#define CONNEX_EXT_SMART        0x01
#define CONNEX_EXT_AP           0x02
#define CONNEX_EXT_ARDUINO      0x04

// Types de motorisation
#define CEC_TYPE_PROPULSION     0x01
#define CEC_TYPE_TRACTION       0x00

// Types de détection de fin de course
#define CEC_TYPE_ARRET_LIGNE    0x01
#define CEC_TYPE_DAMIER         0X02

// Type d'arrêt de course
#define CEC_FIN_DE_COURSE_ZONE  0
#define CEC_FIN_DE_COURSE_LIGNE 1

// Types de départ
#define CEC_TYD_NONE        0         // départ non défini
#define CEC_TYD_IMM         1         // départ immédiat
#define CEC_TYD_PROG        2         // départ programmé
#define CEC_TYD_MAX         3         // Nombre d'états de type de départ

// Type de Fin de course
#define CEC_TYF_NONE        0         // type de fin de course non défini
#define CEC_TYF_FIN_ZONE    1         // fin des zones définies
#define CEC_TYF_LIGNE_FIN   2         // détection ligne de fin
#define CEC_TYF_ARRET_IMM   3         // arrêt immédiat
#define CEC_TYF_DIST_MAX    4         // distance max
#define CEC_TYF_TEMPS_MAX   5         // temps max
#define CEC_TYF_MAX         3         // Nombre d'états de type d'arrivée

// Nombre de segments ou zone
#define CEC_CONFIG_SEGMENTS 10        // Nombre de segments dans la trame config

// Vitesse max de bridage
#define MAX_SPEED_CM_S      1200      // Vitesse maximum en cm/s

// Axes Capteurs
#define CEC_ACCEL_AXE_X     1         // Accelerometre, axe X (longitudinal)
#define CEC_ACCEL_AXE_Y     2         // Accelerometre, axe Y (lateral)
#define CEC_ACCEL_AXE_Z     3         // Accelerometre, axe Z (vertical)

#define CEC_GYRO_AXE_X      1         // Gyrometre, axe X (longitudinal)
#define CEC_GYRO_AXE_Y      2         // Gyrometre, axe Y (lateral)
#define CEC_GYRO_AXE_Z      3         // Gyrometre, axe Z (vertical)

#define CEC_MAGNETO_AXE_X   1         // Magnetometre, axe X (longitudinal)
#define CEC_MAGNETO_AXE_Y   2         // Magnetometre, axe Y (lateral)

// RGB Color Detection Code
#define RGB_BLACK           0x00
#define RGB_BLUE            0x01
#define RGB_GREEN           0x02
#define RGB_CYAN            0x03
#define RGB_RED             0x04
#define RGB_VIOLET          0x05
#define RGB_YELLOW          0x06
#define RGB_WHITE           0x07
#define RGB_NMAX            0x08

// Types d'erreur de transmission
#define CEC_ERR_INTERNAL              -8
#define CEC_ERR_KO_F1                 -7
#define CEC_ERR_FOOTER                -6
#define CEC_ERR_SERIAL_NOT_AVAILABLE  -5
#define CEC_ERR_LENGTH                -4
#define CEC_ERR_WRONG_CMD             -3
#define CEC_ERR_TIMEOUT               -2
#define CEC_ERR                       -1
#define CEC_ERR_OK                     0  // Pas d'erreur
#define CEC_ERR_WRONG_CHECKSUM         1  // Ok mais mauvais checksum
#define CEC_ERR_OK_F0                  2  // Ok avec retour F0
#define CEC_ERR_SEG                    0xFFFF  // erreur segment erroné

// Types de version
#define CEC_VERSION_NONE               0
#define CEC_VERSION_MERE_HARD          1
#define CEC_VERSION_MERE_HARD_MIN      2
#define CEC_VERSION_MERE_SOFT          3
#define CEC_VERSION_MERE_SOFT_MIN      4
#define CEC_VERSION_MOT_HARD           5
#define CEC_VERSION_MOT_HARD_MIN       6
#define CEC_VERSION_MOT_SOFT           7
#define CEC_VERSION_MOT_SOFT_MIN       8
#define CEC_VERSION_CAPT_HARD          9
#define CEC_VERSION_CAPT_HARD_MIN      10
#define CEC_VERSION_CAPT_SOFT          11
#define CEC_VERSION_CAPT_SOFT_MIN      12
#define CEC_VERSION_MAX                13

#define CEC_VERSION_MIN                true
#define CEC_VERSION_MAJ                false

/*******************************************************************************
 * Définition des classes                                                      *
 *******************************************************************************/
// ---- Classe Moteur ---- //
class Moteur
// ~~~~~~~~~
{
  public: 
    typedef struct str_version {
      byte HW_MSB;
      byte HW_LSB;
      byte SW_MSB;
      byte SW_LSB;
    } t_version;

    typedef struct str_versions {
      t_version mere;
      t_version moteur;
      t_version capteur;
    } t_versions;
    
    Moteur();                                   // Constructeur
    // Fonctions de configuration //
    void start(uint32_t serialConfig);          // Démarrage de la liaison avec le moteur
    void setEchoOn();                           // Active la sortie texte vers le terminal
    void setEchoOff();                          // Désactive la sortie texte vers le terminal  
    void purge();                               // Vide le buffer de réception de la liaison moteur
    
    // Fonctions en écriture //
    // Commande Move
    int moveStart(int temps_ms);                                        // Démarrage moteur
    int moveStop(int temps_ms);                                         // Arrêt moteur
    int movePause(int temps_ms);                                        // Pause moteur
    int moveSpeedLimit(float Speed);                                    // Défini la limitation de vitesse en m/s
    int configMoveForward(unsigned int temps_ms,unsigned int Speed);    // Avance à la vitesse moteur
    int configMoveBackward(unsigned int temps_ms,unsigned int Speed);   // Recule à la vitesse moteur
    
    // Fonctions en lecture // 
    int ReadMesures();                          // Trame du dernier ensemble de mesures: réception des valeurs
    unsigned int getMesureNum();                // Récupère le numéro de la mesure
    unsigned int getMesureInst();               // Récupère l'instant 1 de la mesure
    unsigned int getMesureInstant();            // Récupère l'instant 2 de la mesure
    int getMesureAccel(int axe);                // Récupère les valeurs de l'accéléromètre
    int getMesureGyro(int axe);                 // Récupère les valeurs du gyromètre
    int getMesureMagneto(int axe);              // Récupère les valeurs du magnétomètre
    unsigned int getMesureRpm();                // Récupère la vitesse de rotation du pignon moteur
    unsigned int getMesureSpeed();              // Récupère la vitesse à la roue
    unsigned int getMesureDistance();           // Récupère la distance parcourue
    unsigned int getMesureColor();              // Récupère la couleur détectée
    unsigned int getMesureTension();            // Récupère la tension batterie
    unsigned int getMesureCourant();            // Récupère le courant batterie
    unsigned int getMesureTemp();               // Récupère la température
    unsigned int getMesureCourseEtat();         // Recupère l'état de course 
    unsigned int getMesureCourseSegment();      // Recupere le segment de course 
    
    int readInfo();                             // Trame Infos: réception des valeurs
    int getInfoConnex();                        // Récupère l'information de connexion au moteur
    byte getInfoVersion(int type);              // Récupère la version suivant type = CEC_VERSION_xx avec ou sans _MIN
    int printInfoVersion(int type);            // Affiche la version suivant type = CEC_VERSION_xx

    // trame de configuration //
    int initConfig();                           // Trame de configuration: initialisation
    int readConfig();                           // Trame de configuration: réception des valeurs de configuration
    int sendConfig();                           // Trame de configuration: envoie vers le moteur
    unsigned int getConfigLongueurPiste();      // Récupère la longueur de piste
    unsigned int getConfigTailleDamier();       // Récupère la taille du damier
    unsigned int getConfigDiametreRoues();      // Récupère le diamètre des roues
    unsigned int getConfigPignon();             // Récupère le nombre de dent pignon
    unsigned int getConfigCouronne();           // Récupère le nombre de dent Couronne
    unsigned int getConfigVehicule();           // Récupère le type de motorisation
    unsigned int getConfigCourse();             // Récupère le type de fin de course
    unsigned int getConfigPeriodeEch();         // Récupère la periode d'echantillonnage des mesures
    unsigned int getConfigTempsMaxCourse();     // Récupère le temps maximum de la course
    unsigned int getConfigVitesseMin();         // Récupère la vitesse minimale de la course
    unsigned int getConfigVitesseMax();         // Récupère la vitesse maximale de la course
    unsigned int getConfigSegmentVitesse(int numSegment);// Récupère la vitesse finale du segment indiqué de la trame Config
    unsigned int getConfigSegmentTemps(int numSegment);  // Récupère le Temps finale du segment indiqué de la trame Config
    unsigned int getConfigVitesseLente();       // Récupère la vitesse lente de la trame Config
    unsigned int getConfigTimeHeures();         // Récupère les heures du temps actuel de la trame Config
    unsigned int getConfigTimeMinutes();        // Récupère les minutes du temps actuel de la trame Config
    unsigned int getConfigTimeSecondes();       // Récupère les secondes du temps actuel de la trame Config
    unsigned int getConfigTimeMillisecondes();  // Récupère les millisecondes du temps actuel de la trame Config
    void setConfigTimeCourse(float Temps);                   // Défini le temps de course
    void setConfigSpeed(float Speed, float Temps, int zone); // Défini la vitesse pour une zone
    void setConfigDiametreRoues(float DiamRoues);            // Défini le diamètre des roues
    void setConfigLongueurPiste(float LongueurPiste);        // Défini la longueur de piste
    void setConfigMotorisationType(int Choix);               // Défini le type de motorisation
    void setConfigCourseEndType(int EndType);                // Défini le type de fin de course
    
    int ReadBilan();                            // Trame Bilan: réception des valeurs
    unsigned int getBilanFinalTime();           // Récupère le temps final de la course
    unsigned int getBilanFinalDistance();       // Récupère la distance finale de la course
    unsigned int getBilanFinalVmoy();           // Récupère la vitesse moyenne finale de la course
    unsigned int getBilanFinalVmax();           // Récupère la vitesse maximale finale de la course
    unsigned int getBilanFinalCumulCourant();   // Récupère le cumul de courant de la course
    unsigned int getBilanFinalPconso();         // Récupère la puissance consommée de la course
    unsigned int getBilanTypeDepart();          // Récupère le type de départ
    unsigned int getBilanTypeFin();             // Récupère le type d'arrivée
    int getBilanFinalAccelMoy();                // Récupère l'acceleration moyenne finale de la course

    float getBilanAccel();                       // Récupère l'acceleration
    float getBilanVitesse();                     // Récupère la vitesse
    float getBilanPosition();                    // Récupère la position
    byte getBilanCourseEtat();                   // Récupère l'état actuel de course 
    byte getBilanCourseZone();                   // Récupère la zone actuelle de course
    
    // Fonctions annexes // 
    void convertIntToHex(byte *valMSB, byte *valLSB, int valeur); 
    void convertUIntToHex(byte *valMSB, byte *valLSB, unsigned int valeur); 
    void convertHexToInt(byte valMSB, byte valLSB, int *valeur);
    void convertHexToUInt(byte valMSB, byte valLSB, unsigned int *valeur);

#ifndef CODE_LIGHT
    const char* cecSeqCourseEtatTxt[CEC_SEQ_COU_MAX]={
      "Rien",                 // 
      "Attente",              // attente ordre course ou mouvement
      "Demandée",             // initialisation pour une course
      "Calib.",               // calcul
      "Activ.capteur",        // activation carte capteur
      "En cours",             // déroulement de la course
      "Limite",               // dernier pas de la course
      "Ligne d'arrivée",      // pas après passage de la ligne d'arrivèe
      "Fin",                  // fin de la course
      "Annulée",              // annulation
      "Recule",               // initialisation pour le recul de la voiture
      "Avance",               // initialisation pour l'avance de la voiture
      "Positionne",           // positionne la voiture
      "Vit. fixe"             // vitesse fixe 
    };
#endif
  private:
  
    // Variables locales //
    byte octet0, octet1, octet2;
    int valInt;
    unsigned int valUInt;
    float valFloat;

    // Trames //
    // ~~~~~~ //
    
    // Header //
    union uni_header {
      byte car[CEC_LENGTH_HEADER];
      struct {
        byte addr;
        byte start;
        byte lengthMSB;
        byte lengthLSB;
        byte cmd;
      };
    } trameHeader;

    // Footer //
    byte trameFooter[CEC_LENGTH_FOOTER];

    typedef struct{
      byte MSB;
      byte LSB;
    } t_2bytes;

    // Trame de configuration //
    typedef struct {
      t_2bytes  vitesse;
      t_2bytes  temps;
    } t_segment;
    
    typedef struct {
      byte heure;
      byte minute;
      byte seconde;
      t_2bytes milliseconde;
    } t_tempsMot;
    
    union uni_config {
      byte car[CEC_LENGTH_CONFIG];
      struct{
        t_2bytes longueurPiste;
        byte tailleDamier;
        t_2bytes DiametreRoue;
        byte nDentsPignon;
        byte nDentsCouronne;
        byte configVehicule;
        byte configCourse;
        byte reserve1;
        byte periodeEch;
        byte couleurLigDepart;
        byte couleurLigInter1;
        byte couleurLigInter2;
        byte couleurLigFin;
        t_2bytes tempsMaxCourse;
        t_2bytes vitesseMotMin;
        t_2bytes vitesseMotMax;
        t_segment segment[CEC_CONFIG_SEGMENTS];
        t_2bytes VitLente;
        t_tempsMot tempsMot;
      };
    } trameConfig;
    
    const byte trameConfigInit[CEC_LENGTH_CONFIG] =
    // Valeurs par défaut :
    {0x05, 0xDC,                    // 00-01: Longueur de piste = 15m (05DCh)
     0x0A,                          //    02: Taille Damier = 10mm (0Ah)
     0x1C, 0x20,                    // 03-04: Diamètre de roue = 72.00mm (1C20h)
     0x0C,                          //    05: Nombre de dents du pignon = 12 (OCh)
     0x22,                          //    06: Nombre de dents de l'engrenage = 34 (22h)
     0x00,                          //    07: Configuration véhicule = 0 (Propulsion)
     0x00,                          //    08: Configuration course = 0 (arret en fin de zone)
     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 09-14: Réservé
     0x13, 0x88,                    // 15-16: Temps maximal de la course = 5000ms (1388h)
     0x00, 0x50,                    // 17-18: Vitesse Moteur min = 0.8m/s
     0x04, 0xB0,                    // 19-20: Vitesse moteur max = 12.0m/s (non traité)
     0x00, 0x00, 0x00, 0x00,        // 21-24: Segment 0
     0x00, 0x00, 0x00, 0x00,        // 25-28: Segment 1
     0x00, 0x00, 0x00, 0x00,        // 29-32: Segment 2
     0x00, 0x00, 0x00, 0x00,        // 33-36: Segment 3
     0x00, 0x00, 0x00, 0x00,        // 37-40: Segment 4
     0x00, 0x00, 0x00, 0x00,        // 41-44: Segment 5
     0x00, 0x00, 0x00, 0x00,        // 45-48: Segment 6
     0x00, 0x00, 0x00, 0x00,        // 49-52: Segment 7
     0x00, 0x00, 0x00, 0x00,        // 53-56: Segment 8
     0x00, 0x00, 0x00, 0x00,        // 57-60: Segment 9
     0x00, 0x50,                    // 61-62: Vitesse lente = 0.8m/s
     0x00, 0x00, 0x00, 0x00, 0x00}; // 63-67: temps moteur hh:mm:ss:mms
  
    // Trame Move //
    byte trameMove[CEC_LENGTH_MOVE];

    // Trame des mesures //
    typedef union uni_mes_3d {
      uint8_t car[6];
      struct{
        int16_t iMesX;
        int16_t iMesY;
        int16_t iMesZ;
      };
    } st_mes_3d;
    
    typedef union uni_mes_2d {
      uint8_t car[4];
      struct{
        int16_t iMesX;
        int16_t iMesY;
      };
    } st_mes_2d;
    
    typedef struct{
      byte etat;
      byte segment;
    } t_etatCourse;
    
    union uni_mesures {
      byte car[CEC_LENGTH_MESURE];
      struct{
        t_2bytes numMesure;
        t_2bytes instantMesure;
        struct{
          t_2bytes instant;      // instant_mesure
          // capteur ligne
          st_mes_3d accel;            // accéléromètre
          st_mes_3d gyro;             // gyroscope
          st_mes_2d magneto;           // magnétomètre
          t_2bytes rpmMoteur;
          t_2bytes vitesse;           // m/sec
          t_2bytes distance;          // cm
          uint8_t coulLigne;          // couleur ligne
          t_2bytes tensBat;
          t_2bytes courCons;
          t_2bytes temperature;
          t_etatCourse etatCourse;  
        } mes;
      };
    } trameMesures;

    // Trame du bilan course //
    union uni_bilan {
      byte car[CEC_LENGTH_BILAN];
      struct {
        byte tempsCourseMSB;
        byte tempsCourseLSB;
        byte distanceParcourueRoueCmMSB;
        byte distanceParcourueRoueCmLSB;
        byte vitesseMoyenneCm_sMSB;
        byte vitesseMoyenneCm_sLSB;
        byte vitesseMaxiCm_sMSB;
        byte vitesseMaxiCm_sLSB;
        byte consoCourantCumulMSB;
        byte consoCourantCumulLSB;
        byte puissanceConsoMSB;
        byte puissanceConsoLSB;
        byte typeDepart;
        byte motifFinCourse;
        byte accelerationMoyenneMSB;
        byte accelerationMoyenneLSB;
        byte accelerationAcceleroMSB;
        byte accelerationAcceleroLSB;
        byte vitesseAcceleroMSB;
        byte vitesseAcceleroLSB;
        byte positionAcceleroMSB;
        byte positionAcceleroLSB;
        byte seqCourse;
      };
    } trameBilan;
    
    //Temps max course, par défaut = 5,000sv //
    byte DureeCourseMSB = 0x13;
    byte DureeCourseLSB = 0x88;
    // Diametre des roues, par défault 73,9mm //
    byte DiametreRouesMSB = 0x02;
    byte DiametreRouesLSB = 0xE3;
    // Longueur de piste, par défault 15,00m //
    byte LongueurPisteMSB = 0x05;
    byte LongueurPisteLSB = 0xDC;
    // Tyme de motorisation, par défaut 0 (0:propulsion, 1:traction) //
    int MotorisationType = 0;
    // Type de fin de course, par défaut 0 (0:fin de zone, 1:ligne d'arrivée) //
    int CourseEnd = 0;

    // Trame de données d'information moteur //
    union uni_info {
      byte car[CEC_LENGTH_INFO];
      struct {
        byte vers_mere_hard_maj;
        byte vers_mere_hard_min;
        byte vers_mere_soft_maj;
        byte vers_mere_soft_min;
        byte vers_mot_hard_maj;
        byte vers_mot_hard_min;
        byte vers_mot_soft_maj;
        byte vers_mot_soft_min;
        byte vers_capt_hard_maj;
        byte vers_capt_hard_min;
        byte vers_capt_soft_maj;
        byte vers_capt_soft_min;
        byte trame_format;
        byte heure;
        byte minute;
        byte seconde;
        byte millisecondeMSB;
        byte millisecondeLSB;
        byte etatConnexCarte;
        byte etatConnexExt;
        byte vBatMSB;
        byte vBatLSB;
        byte codeErreur;
      };
      t_versions versions;
    } trameInfo;
    
    // Variables temporelles, initalisation au temps de démarrage du programme //
    unsigned long tempsEcoule = millis();
    unsigned long tempsDebut = millis();

    // Variables booléennes (flags) //
    bool bEchoOn = true;
    bool bTrameConfigOK = false;
    bool bTrameMesuresOK = false;
    bool btrameBilanOK = false;
    bool bTrameInfoOK = false;    
    bool bCheckSum = false;
    // Variables
    uint16_t checkSum = 0;          // Somme de vérification
    uint16_t longueurData = 0;      // Longueur des données
    byte sendCmd = 0;               // Valeur de la commande moteur envoyée
    byte rcvCmd = 0;                // Valeur de la commande moteur reçue

    // Methodes privées //
    int writeHeader(byte cmd, uint16_t size, bool bWithCheckSum); // Ecriture entête message
    int writeFooter();                                            // Ecriture fin de message
    int readHeader();                                             // Lecture entête message
    int calculCrc(uint16_t dataSize, byte* ptData);                  // Calcule le crc du message avec header et footer
    int sendTrame(byte cmd, byte* ptData, uint16_t dataSize, bool bWithCheckSum);                  // Envoi entête + données + fin message
    int readTrame(byte cmd, byte *ptData, uint16_t dataSize, bool bWithCheckSum); 
    int readData(uint16_t nData, byte* ptData);                   // Lecture données message
    int readFooter();                                             // Lecture fin de message
    void printBuffer(byte *buff, uint16_t dataSize);              // Sortie texte du buffer en hexadecimal 
    void printHeader();                                           // Sortie texte du buffer header en hexadecimal       
    void printFooter();                                           // Sortie texte du buffer footer en hexadecimal           
};  // class Moteur

/* Variables globales */ 

#endif  // CEC_LIB_H
